import ij.*;
import ij.plugin.PlugIn;
import ij.plugin.Duplicator;
import ij.process.*;
import ij.measure.*;
import ij.gui.GenericDialog;
import java.awt.*;

/* Converts a 4D hyperstack to a time series of Z-stack montages, and autoscales the channel displays. */
public class Make_Montage_Series implements PlugIn {
  
	private ImagePlus originalImage, scaledImage;
	private ImageProcessor scaled;
	
	private int width, height, scaledWidth, scaledHeight, slices, frames, channels;   // Hyperstack parameters.
	private int first, last, trimmedSlices;                                           // User-chosen slices.
	private int maxWidth, maxHeight;                                                  // Maximum size of montage.
	private double scaleFactor;
	private boolean validScaleFactor = false;
	private int blackWindowWidth, blackWindowHeight, columns, rows;                   // Montage dimensions.
	private String title;
	
	private static final int XCORNER = 10, YCORNER = 100, GAP = 10, AUTO_THRESHOLD = 5000;
	
	//------------------------------------------------------------------------------------------------------------------------  
	
    public void run(String arg) {
      
      IJ.resetEscape();

      IJ.run("Open...");
      originalImage = IJ.getImage();
      title = originalImage.getTitle();
 
      if ( !(originalImage.getBitDepth() == 8)  || !(title.endsWith(".tif") || title.endsWith(".TIF")) || 
          !originalImage.isHyperStack() || !(originalImage.getNSlices() > 1) ) {
        IJ.showMessage("This plugin requires an 8-bit TIFF 4D hyperstack.");
        originalImage.close();
        return;
      }
      
      title = title.substring(0, title.length() - 4);           // Remove the extension.
      
      width = originalImage.getWidth();
      height = originalImage.getHeight();
      slices = originalImage.getNSlices();
      frames = originalImage.getNFrames();
      channels = originalImage.getNChannels();
      
      Dimension screen = IJ.getScreenSize();
      maxWidth = screen.width - 2 * XCORNER - 10;               // 5-px borders on each side of the image window.
      maxHeight = screen.height - YCORNER - XCORNER - 85;       // Takes into account the title bar and scroll bars.
      
      double provisionalScaleFactor = 4.0;                      // Default scale factor.
      first = 1;
      last = slices;
      
      while (!validScaleFactor) {
        GenericDialog gd = new GenericDialog("Parameters");
        gd.addNumericField("Scale Factor:", provisionalScaleFactor, 1);
        gd.addNumericField("First Slice:", first, 0);
        gd.addNumericField("Last Slice:", last, 0);
        gd.showDialog();
        if (gd.wasCanceled()) return;
        
        scaleFactor = gd.getNextNumber();
        scaledWidth = (int) Math.round(width * scaleFactor);
        scaledHeight = (int) Math.round(height * scaleFactor);
        
        first = (int) gd.getNextNumber();
        last = (int) gd.getNextNumber();
        if (first < 1 || last > slices || last < first) {
          IJ.showMessage("Those slice numbers are not valid.");
          originalImage.close();
          return;
        }
        trimmedSlices = last - first + 1;                        // Actual number of slices used in the montage.
        
        // Calculate how many scaled images will fit horizontally and vertically, separated by GAP pixels.
        columns = (int) Math.floor((double) (maxWidth - GAP) / (double) (scaledWidth + GAP));
        columns = Math.min(columns, trimmedSlices);
        blackWindowWidth = columns * (scaledWidth + GAP) + GAP;
        rows = (int) Math.ceil((double) trimmedSlices / (double) columns);
        blackWindowHeight = rows * (scaledHeight + GAP) + GAP;
        if (blackWindowHeight > maxHeight) {
          IJ.showMessage("The images will not fit on the screen with that scale factor.");
          provisionalScaleFactor = findMaxScaleFactor();
        }
        else {
          validScaleFactor = true;
        }
      }
      
      // Create a scaled version of the original image.
      scaledImage = new Duplicator().run(originalImage);
      scaled = scaledImage.getProcessor();
      
      originalImage.close();
         
      if (scaleFactor != 1.0) {
        scaled.setInterpolationMethod(ImageProcessor.BICUBIC);      // BILINEAR is faster, BICUBIC is better.
        StackProcessor sp = new StackProcessor(scaledImage.getStack(), scaled);
        ImageStack s2 = sp.resize(scaledWidth, scaledHeight);
        scaledImage.setStack(null, s2);
      }
      ImageStack scaledStack = scaledImage.getStack();
      
      // Make a black background window for the montage.
      ImagePlus compositeMontage = makeCompositeMontage(title, blackWindowWidth, blackWindowHeight, 3, frames);
      ImageProcessor composite = compositeMontage.getProcessor();
      compositeMontage.show();
   
      int slice;
      ImageProcessor ipSlice;
      
      // Make composite montage.
      int xPos, yPos;
      for (int t = 1; t <= frames; t++) {
        for (int ch = 1; ch <= 3; ch++) {
          if (IJ.escapePressed()) {
            compositeMontage.close();
            IJ.showStatus("Plugin aborted.");
            return;
          }
          compositeMontage.setPosition(ch, 1, t);
          if (channels == 3) {                                        // Red and green images.
            int firstSlice = 3 * slices * (t - 1) + ch;               // First slice in a Z-stack for channel ch and time t.
            for (int z = first; z <= last; z++) {
              slice = firstSlice + 3 * (z - 1);
              ipSlice = scaledStack.getProcessor(slice);
              xPos = GAP + ((z - first) % columns) * (scaledWidth + GAP);
              yPos = GAP + ((z - first) / columns) * (scaledHeight + GAP);
              composite.insert(ipSlice, xPos, yPos);
            }
          }
          else if (channels == 2) {                                   // Green images only.
            if (ch > 1) {                                             // Skip the red channel.
              int firstSlice = 2 * slices * (t - 1) + (ch  - 1);      // First slice in a Z-stack for channel ch and time t.
              for (int z = first; z <= last; z++) {
                slice = firstSlice + 2 * (z - 1);
                ipSlice = scaledStack.getProcessor(slice);
                xPos = GAP + ((z - first) % columns) * (scaledWidth + GAP);
                yPos = GAP + ((z - first) / columns) * (scaledHeight + GAP);
                composite.insert(ipSlice, xPos, yPos);
              }
            }
          }
        }
      }
      
      scaledImage.close();
                                 
      for (int ch = 4 - channels; ch <= 3; ch++) {                    // Start at 1 if red and green, or at 2 if green only.
        compositeMontage.setPosition(ch,1,1);
        channelAdjust(compositeMontage);
        compositeMontage.updateAndDraw();
      }
      
      compositeMontage.setPosition(1,1,1);
      
      // Record image data that will be used later to regenerate a hyperstack.
      String compositeMontageInfo =     "slices: " + trimmedSlices + "\n" + 
                                        "columns: " + columns + "\n" +
                                        "rows: " + rows + "\n" +
                                        "GAP: " + GAP + "\n" +
                                        "slice width: " + scaledWidth + "\n" +
                                        "slice height: " + scaledHeight;
      compositeMontage.setProperty("Info", compositeMontageInfo);
      
      compositeMontage.changes = true;
      
	}

    //========================================================================================================================

    /* Creates a black hyperstack composite image window for displaying a multi-channel montage time series. */
    private ImagePlus makeCompositeMontage(String title, int windowWidth, int windowHeight, int channels, int frames) {
      ImagePlus blackImage = IJ.createImage(null, "8-bit black", windowWidth, windowHeight, 1);
      ImageProcessor black = blackImage.getProcessor();
      
      ImageStack xycztStack = new ImageStack(windowWidth, windowHeight);
      for (int ch = 1; ch <= channels; ch++) {
        for (int t = 1; t <= frames; t++) {
          xycztStack.addSlice(null, black.duplicate());
        }
      }
      
      ImagePlus compositeMontage = new ImagePlus(title + " Montage.tif", xycztStack);
      compositeMontage.setDimensions(channels, 1, frames);
      compositeMontage = new CompositeImage(compositeMontage, CompositeImage.COMPOSITE);
      
      compositeMontage.setOpenAsHyperStack(true);
      
      return compositeMontage;
    }
    
    //========================================================================================================================
    
    /* Automatically adjusts the maximum threshold for the currently selected channel of the image. */
    private void channelAdjust(ImagePlus imp) {
      // The display maximum value is set to the highest number at which the fraction of pixels having at least the
      // display maximum value exceeds 1/autoThreshold. Lowering autoThreshold results in more saturated pixels.
      Calibration cal = imp.getCalibration();
      imp.setCalibration(null);
      ImageStatistics stats = imp.getStatistics();          // Get uncalibrated statistics.
      imp.setCalibration(cal);
      
      int[] histogram = stats.histogram;
      int cutoff = stats.pixelCount/AUTO_THRESHOLD;
      boolean found = false;
      int i = 256;
      while (!found) {
        i--;
        found = ((histogram[i] > cutoff) || (i == 0));
      }
      int max = i;
      imp.setDisplayRange(0, max);
      
    }
    
    //========================================================================================================================
    
    /* Determine how large the scale factor can be without exceeding the screen size. */
    private double findMaxScaleFactor() {
      double testScaleFactor;
      for (int i = 40; i >= 10; i--) {
        testScaleFactor = i / 10.0;
        scaledWidth = (int) Math.round(width * testScaleFactor);
        scaledHeight = (int) Math.round(height * testScaleFactor);
        
        // Calculate how many scaled images will fit horizontally and vertically, separated by GAP pixels.
        columns = (int) Math.floor((double) (maxWidth - GAP) / (double) (scaledWidth + GAP));
        columns = Math.min(columns, trimmedSlices);
        blackWindowWidth = columns * (scaledWidth + GAP) + GAP;
        rows = (int) Math.ceil((double) trimmedSlices / (double) columns);
        blackWindowHeight = rows * (scaledHeight + GAP) + GAP;
        if (blackWindowHeight <= maxHeight) {
          return testScaleFactor;
        }
      }
      
      return 1.0;                                           // Fallback option.
    }
    
}
